/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Id: Main.c 3194 2011-12-14 16:36:28Z Robert $
   Last Modification:
    $Author: Robert $
    $Date: 2011-12-14 17:36:28 +0100 (Mi, 14 Dez 2011) $
    $Revision: 3194 $

   Targets:
     linux    : yes

   Description:
    Serial connector (USB/RS232) example

   Changes:

     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
      2        01.03.18    SD       - Serial_IF_Send: continue interrupted sends
                                    - added support for netX52
      1        23.01.14    SD       initial version

**************************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <termios.h>
#include <unistd.h>
#include <string.h>
#include <pthread.h>
#include <fcntl.h>
#include <dirent.h>
#include <poll.h>
#include <errno.h>

#include "ConnectorAPI.h"
#include "OS_Dependent.h"
#include "netXTransport.h"
#include "cifXErrors.h"

/* global defines */
#ifdef TRANSPORT_DEBUG
    #define DEBUG printf
#else
    #define DEBUG //
#endif

#define RECV_BUFFER_SIZE         1048  /* size of receivce buffer                 */
#define INTERFACE_EVENT_ATTACHED 0     /* event definition                        */
#define INTERFACE_EVENT_DETACHED 1     /* event definition                        */

#define PARSER_BUFFER_SIZE       256   /* required for parsing configuration file */

/* USB identifier */
#define HILSCHER_VENDOR_ID 0x1939

#define PRODUCT_NETTAP     0x001D
#define PRODUCT_UNDEF      0x000F
#define PRODUCT_NETX52     0x0200

#define DEV_CLASS_NETX100  0x0001
#define DEV_CLASS_CDC      0x0002

/* global variables */
static int              g_fConnectorAdded = 0;
static NETX_CONNECTOR_T s_tSerConnector   = {{0}};

TAILQ_HEAD(CONNECTOR_LIST_T, SER_IF_DATA_Ttag);

/* connector instance */
typedef struct SERIAL_CONNECTOR_INFO_Ttag
{
  uint32_t                ulConfigCount;  /* number registered connector interfaces */
  struct CONNECTOR_LIST_T tConnectorList; /* list of connector interfaces           */

} SERIAL_CONNECTOR_INFO_T;

/* structure definitions           */
/* connector information structure */
typedef struct SER_IF_DATA_Ttag
{
  TAILQ_ENTRY(SER_IF_DATA_Ttag) tList;

  char                    szInterfaceName[NXT_MAX_PATH]; /* Interface name                          */
  char                    szDevInternalName[256];
  int                     iDeviceNum;
  char                    szPortIdent[256];
  int                     iDevFile;
  struct termios          old_config;
  pthread_t               tRXThread;                     /* Thread info of receive thread           */
  void*                   pvRXThreadStopped;             /* Event for receive thread termination    */
  pthread_t               tEventThread;                  /* Thread info of event handler thread     */
  int                     fDetached;
  int                     fThreadRunning;

  uint32_t                ulState;                       /* State of the interface                  */

  uint32_t                ulEvent;                       /* Event that occurs                       */
  void*                   pvConnectorLock;

  PFN_NETXCON_DEVICE_RECEIVE_CALLBACK   pfnRxCallback;        /* Receive data Callback                            */
  void*                                 pvRxCallbackData;     /* User data (for receive data callback)            */
  PFN_NETXCON_INTERFACE_NOTIFY_CALLBACK pfnDevNotifyCallback; /* Interface state change Callback                  */
  void*                                 pvUser;               /* User parameter (interface state change callback) */

} SER_IF_DATA_T, *PSER_IF_DATA_T;


/* function declaration (see ConnectorAPI.h) */
int32_t APIENTRY Serial_GetIdentifier    ( char* szIdentifier, NETX_TRANSPORT_UUID_T* pvUUID);
int32_t APIENTRY Serial_Open             ( PFN_NETXCON_INTERFACE_NOTIFY_CALLBACK pfnDevNotifyCallback, void* pvUser);
int32_t APIENTRY Serial_Close            ( void);
void*   APIENTRY Serial_CreateIF         ( const char* szDeviceName);
int32_t APIENTRY Serial_GetInformation   ( NETX_CONNECTOR_INFO_E eCmd, uint32_t ulSize, void* pvConnectorInfo);
int32_t APIENTRY Serial_GetConfig        ( NETX_CONNECTOR_CONFIG_CMD_E eCmd, void* pvConfig);
int32_t APIENTRY Serial_SetConfig        ( NETX_CONNECTOR_CONFIG_CMD_E eCmd, const char* szConfig);
int32_t APIENTRY Serial_IF_Start         ( void* pvInterface, PFN_NETXCON_DEVICE_RECEIVE_CALLBACK pfnReceiveData, void* pvUser);
int32_t APIENTRY Serial_IF_Stop          ( void* pvInterface);
int32_t APIENTRY Serial_IF_Send          ( void* pvInterface, unsigned char* pabData, uint32_t ulDataLen);
int32_t APIENTRY Serial_IF_GetInformation( void* pvInterface, NETX_INTERFACE_INFO_E eCmd, uint32_t ulSize, void* pvInterfaceInfo);

/* internal helper */
static void*     CommThread      ( void* pvParam);
static void*     EventThread     ( void* pvParam);
static int       GetConfigString ( const char* szFile, const char* szKey, char** szValue);
static void      DetectNetxDevice( void);
static int       ConfigureIF     ( int fd);

static SERIAL_CONNECTOR_INFO_T s_tSerConnectorInfo;

#ifdef CONNECTION_CONTROL
#include <libudev.h>

pthread_t s_ControlThread;

void* ConnectorControlThread(void* pvParam);

#define SUBSYSTEM "usb"

int s_run_control_thread = 0;

static void process_device(struct udev_device* dev) {
  int activate = -1;

  if (dev) {
    const char*    name            = udev_device_get_devpath(dev);
    const char*    action          = udev_device_get_action(dev);
    const char*    vendor          = udev_device_get_sysattr_value(dev, "idVendor");
    PSER_IF_DATA_T ptInterfaceData = NULL;

    if (! action)
      action = "exists";

    if (strcmp(action, "exists") == 0) {
      if ((vendor != NULL) && (strcmp(vendor,"1939") == 0))
        activate = 0;
    } else if (strcmp(action, "add") == 0) {
      if ((vendor != NULL) && (strcmp(vendor,"1939") == 0))
        activate = 1;
    } else if (strcmp(action, "remove") == 0) {
      activate = 2;
    }

    TAILQ_FOREACH( ptInterfaceData, &s_tSerConnectorInfo.tConnectorList, tList) {
      char tmp[512];

      snprintf(tmp, 511, "/sys%s", name);
      if (strncmp( ptInterfaceData->szPortIdent, tmp, strlen(tmp)) == 0) {
        if (activate == 1) {
          ptInterfaceData->pfnDevNotifyCallback( ptInterfaceData->szInterfaceName, eNXT_DEVICE_ATTACHED, ptInterfaceData, ptInterfaceData->pvUser);
        } else if (activate == 2) {
          ptInterfaceData->pfnDevNotifyCallback( ptInterfaceData->szInterfaceName, eNXT_DEVICE_DETACHED, ptInterfaceData, ptInterfaceData->pvUser);
        }
      }
    }
    udev_device_unref(dev);
  }
}

void* ConnectorControlThread(void* pvParam) {
  struct udev*         udev = udev_new();
  struct udev_monitor* mon  = NULL;
  int                  fd   = 0;

  if (!udev) {
    fprintf(stderr, "Serial-con: udev_new() failed. Connector control not available!\n");
    return NULL;
  }
  mon = udev_monitor_new_from_netlink(udev, "udev");
  fd = udev_monitor_get_fd(mon);

  if ((mon != NULL) && ((fd = udev_monitor_get_fd(mon)) >= 0)) {
    udev_monitor_filter_add_match_subsystem_devtype(mon, SUBSYSTEM, NULL);
    udev_monitor_enable_receiving(mon);

    while (s_run_control_thread) {
      struct timeval timeout = { 0, 1000*500 /*100ms*/};
      int            ret     = 0;
      fd_set         fds;

      FD_ZERO(&fds);
      FD_SET(fd, &fds);

      if ((ret = select(fd+1, &fds, NULL, NULL, &timeout)) < 0) {
        fprintf(stderr, "Serial-con: Failed to monitor connector (%d). Connector control not available!\n", errno);
        break;
      }
      if (FD_ISSET(fd, &fds)) {
        struct udev_device* dev = udev_monitor_receive_device(mon);
        process_device(dev);
      }
    }
  } else {
    fprintf(stderr, "Serial-con: Failed to start connector monitoring. Connector control not available!\n");
  }
  udev_unref(udev);
  return NULL;
}
#endif

/*****************************************************************************/
/*! Connector initialization and registration
*   \return 0 on success                                                     */
/*****************************************************************************/
int32_t Serial_Connector_Init(void)
{
  int32_t lRet = NXT_NO_ERROR;

  /* initialize function pointer table */
  memset(&s_tSerConnector, 0, sizeof(s_tSerConnector));
  s_tSerConnector.tFunctions.pfnConGetIdentifier       = Serial_GetIdentifier;
  s_tSerConnector.tFunctions.pfnConOpen                = Serial_Open;
  s_tSerConnector.tFunctions.pfnConClose               = Serial_Close;
  s_tSerConnector.tFunctions.pfnConCreateInterface     = Serial_CreateIF;
  s_tSerConnector.tFunctions.pfnConGetInformation      = Serial_GetInformation;
  s_tSerConnector.tFunctions.pfnConGetConfig           = Serial_GetConfig;
  s_tSerConnector.tFunctions.pfnConSetConfig           = Serial_SetConfig;

  s_tSerConnector.tFunctions.pfnConIntfStart           = Serial_IF_Start;
  s_tSerConnector.tFunctions.pfnConIntfStop            = Serial_IF_Stop;
  s_tSerConnector.tFunctions.pfnConIntfSend            = Serial_IF_Send;
  s_tSerConnector.tFunctions.pfnConIntfGetInformation  = Serial_IF_GetInformation;

  s_tSerConnector.tFunctions.pfnConCreateDialog        = NULL;
  s_tSerConnector.tFunctions.pfnConEndDialog           = NULL;

  /* add usb/rs232 connector interface */
  if (NXT_NO_ERROR == (lRet = netXTransportAddConnector(&s_tSerConnector)))
  {
    DEBUG("Serial-con: USB/rs232 connector successfully added\n");
    g_fConnectorAdded = 1;
  }
  return lRet;
}

int32_t connector_init(void) {
  return Serial_Connector_Init();
}

/*****************************************************************************/
/*! De-registers connector                                                   */
/*****************************************************************************/
void Serial_Connector_Deinit(void)
{
  if (g_fConnectorAdded == 1)
  {
    /* remove connector interface */
    netXTransportRemoveConnector(&s_tSerConnector);
  }
}

void connector_deinit(void) {
  Serial_Connector_Deinit();
}

/*****************************************************************************/
/*! Returns the UU-Id                                                        */
/*****************************************************************************/
int32_t Serial_GetIdentifier(char* szIdentifier, NETX_TRANSPORT_UUID_T* ptUUID)
{
  snprintf( szIdentifier, strlen("serial") + 1, "serial"); /* return anything different than the tcp connector example */
  snprintf( (char*)ptUUID, strlen("serial") + 1, "serial");

  return 0;
}

/*****************************************************************************/
/*! Retrieve connector configuration and initializes connector resources
*   \param pfnDevNotifyCallback Callback (interface detection)
*   \param pvUser               User data
*   \return 0 on success                                                     */
/*****************************************************************************/
int32_t APIENTRY Serial_Open( PFN_NETXCON_INTERFACE_NOTIFY_CALLBACK pfnDevNotifyCallback, void* pvUser)
{
  int32_t        lRet            = NXT_NO_ERROR;
  PSER_IF_DATA_T ptInterfaceData = NULL;

#ifdef CONNECTION_CONTROL
  s_run_control_thread = 1;
  pthread_create( &s_ControlThread, NULL, ConnectorControlThread, NULL);
#endif

  TAILQ_INIT(&s_tSerConnectorInfo.tConnectorList);

  /* enumerate over all interface and check for a netX device */
  DetectNetxDevice();

  /* notify every detected interface */
  TAILQ_FOREACH( ptInterfaceData, &s_tSerConnectorInfo.tConnectorList, tList)
  {
    ptInterfaceData->pvConnectorLock      = OS_CreateLock();
    ptInterfaceData->pvRXThreadStopped    = OS_CreateEvent();
    ptInterfaceData->pvUser               = pvUser;
    ptInterfaceData->pfnDevNotifyCallback = pfnDevNotifyCallback;

    if ( (ptInterfaceData->pvConnectorLock == NULL) ||
         (ptInterfaceData->pvRXThreadStopped == NULL)) {

      if (ptInterfaceData->pvConnectorLock)
        OS_DeleteLock(ptInterfaceData->pvConnectorLock);

      ptInterfaceData->pvConnectorLock = NULL;

      DEBUG( "Serial-con: Error allocating required resources for interface %s\n", ptInterfaceData->szInterfaceName);
    } else {
      /* Notify netXTransport-Layer that new interface is created */
      pfnDevNotifyCallback( ptInterfaceData->szInterfaceName, eNXT_DEVICE_ATTACHED, ptInterfaceData, pvUser);
    }
  }
  return lRet;
}

/*****************************************************************************/
/*! Closes connector (and all related interfaces)
*   \return 0 success                                                        */
/*****************************************************************************/
int32_t APIENTRY Serial_Close( void)
{
  PSER_IF_DATA_T ptInterfaceData = NULL;

#ifdef CONNECTION_CONTROL
  if (s_run_control_thread != 0) {
    s_run_control_thread = 0;
    pthread_join( s_ControlThread, NULL);
  }
#endif
  while(NULL != (ptInterfaceData = TAILQ_FIRST( &s_tSerConnectorInfo.tConnectorList)))
  {
    TAILQ_REMOVE( &s_tSerConnectorInfo.tConnectorList, ptInterfaceData, tList);

    ptInterfaceData->pfnDevNotifyCallback( ptInterfaceData->szInterfaceName, eNXT_DEVICE_DELETED, ptInterfaceData, ptInterfaceData->pvUser);

    /* check if we have to stop the interface */
    if (ptInterfaceData->pvConnectorLock != NULL)
    {
      OS_EnterLock( ptInterfaceData->pvConnectorLock);
      if (ptInterfaceData->ulState != eINTERFACE_STATE_STOPPED)
      {
        Serial_IF_Stop( ptInterfaceData);
      }
      OS_LeaveLock( ptInterfaceData->pvConnectorLock);
    }
    if (ptInterfaceData->pvRXThreadStopped != NULL)
      OS_DeleteEvent( ptInterfaceData->pvRXThreadStopped);

    free( ptInterfaceData);
    ptInterfaceData = NULL;

    s_tSerConnectorInfo.ulConfigCount--;
  }
  return 0;
}

/*****************************************************************************/
/*! Create interface and initializes connection
*   \param szDeviceName  Name of the interface
*   \return Pointer to the interface                                         */
/*****************************************************************************/
void* APIENTRY Serial_CreateIF(const char* szDeviceName)
{
  PSER_IF_DATA_T ptInterfaceData = NULL;

  TAILQ_FOREACH( ptInterfaceData, &s_tSerConnectorInfo.tConnectorList, tList)
  {
    if (strcmp( ptInterfaceData->szInterfaceName, szDeviceName) == 0)
    {
      return ptInterfaceData;
    }
  }
  return NULL;
}

/*****************************************************************************/
/*! Send data
*   \param pvInterface  Pointer to interface info structure
*   \param pabData      Pointer to the data to send
*   \param pvInterface  Length of buffer pabData
*   \return 0 on success                                                     */
/*****************************************************************************/
int32_t APIENTRY Serial_IF_Send( void* pvInterface, unsigned char* pabData, uint32_t ulDataLen)
{
  PSER_IF_DATA_T ptInterfaceData = (PSER_IF_DATA_T)pvInterface;
  int            iResult         = 0;
  int32_t        lRet            = NXT_NO_ERROR;

  do
  {
    if ((iResult = write( ptInterfaceData->iDevFile, pabData, ulDataLen)) > 0) {
      pabData   += iResult;
      ulDataLen -= iResult;
    }
  } while ((iResult >= 0) && (ulDataLen > 0));

  if (ulDataLen != 0) {
    lRet = CIFX_TRANSPORT_ABORTED;
    DEBUG("Serial-Con: transmit failed (%d)\n", errno);
  }
  return lRet;
}


/*****************************************************************************/
/*! Thread processes receive data
*   \param pvParam  User data (pointer to interface info structure)
*   \return NULL                                                             */
/*****************************************************************************/
void* CommThread(void* pvParam)
{
  PSER_IF_DATA_T ptInterfaceData = (PSER_IF_DATA_T)pvParam;
  int            iResult         = 0;
  unsigned char  abBuffer[RECV_BUFFER_SIZE];
  struct pollfd  fds;

  fds.fd     = ptInterfaceData->iDevFile;
  fds.events = POLLIN | POLLPRI;

  while(ptInterfaceData->fThreadRunning)
  {
    if (0 < (iResult = poll( &fds, 1, 10)))
    {
      uint32_t ulDataRead = 0;

      if (fds.revents & POLLERR) {
        /* NOTE: "Device disappeared! Might be reset process... */
        /* relax the cpu a little */
        OS_Sleep(10);
        continue;
      }

      while (0<(ulDataRead = read( ptInterfaceData->iDevFile, abBuffer, RECV_BUFFER_SIZE)))
      {
        ptInterfaceData->pfnRxCallback( abBuffer, ulDataRead, ptInterfaceData->pvRxCallbackData);
      }
    } else if (iResult == 0)
    {
      /* timeout */
    } else
    {
      /* error during poll */
    }
  }
  if (NULL != ptInterfaceData->pvRXThreadStopped)
    OS_SetEvent( ptInterfaceData->pvRXThreadStopped);

  return NULL;
}

/*****************************************************************************/
/*! Returns connector information (currently not used)
*   \param eCmd             Command to execute
*   \param ulSize           Size of buffer pvConnectorInfo
*   \param pvConnectorInfo  Pointer to buffer
*   \return 0 on success                                                     */
/*****************************************************************************/
int32_t APIENTRY Serial_GetInformation( NETX_CONNECTOR_INFO_E eCmd, uint32_t ulSize, void* pvConnectorInfo)
{
  int32_t lRet = NXT_NO_ERROR;

  UNREFERENCED_PARAMETER(eCmd);
  UNREFERENCED_PARAMETER(ulSize);
  UNREFERENCED_PARAMETER(pvConnectorInfo);

  return lRet;
}

/*****************************************************************************/
/*! Returns connector configuration (currently not used)
*   \param eCmd      Command to execute
*   \param pvConfig  Pointer to buffer
*   \return 0 on success                                                     */
/*****************************************************************************/
int32_t APIENTRY Serial_GetConfig( NETX_CONNECTOR_CONFIG_CMD_E eCmd, void* pvConfig)
{
  int32_t lRet = NXT_NO_ERROR;

  UNREFERENCED_PARAMETER(eCmd);
  UNREFERENCED_PARAMETER(pvConfig);

  return lRet;
}

/*****************************************************************************/
/*! Sets connector configuration (currently not used)
*   \param eCmd      Command to execute
*   \param pvConfig  Pointer to buffer
*   \return 0 on success                                                     */
/*****************************************************************************/
int32_t APIENTRY Serial_SetConfig( NETX_CONNECTOR_CONFIG_CMD_E eCmd, const char* szConfig)
{
  int32_t lRet = NXT_NO_ERROR;

  UNREFERENCED_PARAMETER(eCmd);
  UNREFERENCED_PARAMETER(szConfig);

  return lRet;
}

/*****************************************************************************/
/*! Starts interface (Com-Thread)
*   \param pvInterface     Pointer to the interface info structure
*   \param pfnReceiveData  Callback (receive data callback)
*   \param pvUser          User data
*   \return 0 on success                                                     */
/*****************************************************************************/
int32_t APIENTRY Serial_IF_Start( void* pvInterface, PFN_NETXCON_DEVICE_RECEIVE_CALLBACK pfnReceiveData, void* pvUser)
{
  PSER_IF_DATA_T ptInterfaceData     = (PSER_IF_DATA_T)pvInterface;
  pthread_attr_t polling_thread_attr = {{0}};
  int            iTmpRet             = 0;
  int32_t        iRet                = -1;
  char           szTmpPortIdent[256];
  char           szTmpPath[278];
  char           szDevFilePath[272];
  size_t         tPathLen            = 1;
  int32_t        lRet                = -1;
  int            iMaxCount           = 0;
  struct dirent**  pptNamelist = NULL;

  /* check whether it is a rs232 or usb connection */
  if (0 != strncmp("ttyS", ptInterfaceData->szDevInternalName, strlen(ptInterfaceData->szDevInternalName)))
  {/* USB: */
    /* to ensure usb hotplug we have to iterate over all interfaces                   */
    /* a usb device may change its name (depending on the order they were plugged in) */
    /* so iterate over all interface and compare the internal port identification     */
    int tmp = iMaxCount = scandir("/sys/class/tty", &pptNamelist, 0, alphasort);
    while(iMaxCount-->0)
    {
      //TODO: check if deactivated or free (block it when accessed)
      char tmpName[512];
      sprintf( tmpName, "%s%d", ptInterfaceData->szDevInternalName, ptInterfaceData->iDeviceNum);

      if (0 == strncmp( pptNamelist[iMaxCount]->d_name, tmpName, strlen(tmpName)))
      {
        /* we have found a adequate interface */
        DEBUG("Serial-con: CDT USB device detected %s\n", pptNamelist[iMaxCount]->d_name);

        snprintf( szTmpPath, 278, "/sys/class/tty/%s", pptNamelist[iMaxCount]->d_name);

        if (NULL != realpath( szTmpPath, szTmpPortIdent))
        {
          tPathLen = lRet;

          if (0 == strncmp( ptInterfaceData->szPortIdent, szTmpPortIdent, tPathLen))
          {
            DEBUG("Serial-con: CDT USB device on well known port for interface %s\n", ptInterfaceData->szInterfaceName);

            /* get new device number */
            //sscanf(pptNamelist[iMaxCount]->d_name, "%*[^0-9]%d", &ptInterfaceData->iDeviceNum);

            snprintf( szDevFilePath, 272, "/dev/%s%d", ptInterfaceData->szDevInternalName, ptInterfaceData->iDeviceNum);

            DEBUG("Serial-con: CDT USB device now estimated under %s\n", szDevFilePath);

            ptInterfaceData->iDevFile = open( szDevFilePath, O_RDWR, O_NONBLOCK);

            if (ptInterfaceData->iDevFile <0)
              ptInterfaceData->iDevFile = -errno;
            /* Somehow we need to configure once the serial connection of a USB CDC connection to netX52... */
            ConfigureIF( ptInterfaceData->iDevFile);
            break;
          }
        }
      }
    }
    while (tmp>0) {
      tmp--;
      OS_Memfree(pptNamelist[tmp]);
    }
    if (pptNamelist != NULL)
      OS_Memfree(pptNamelist);

    if (ptInterfaceData->iDevFile < 0)
    {
      /* error opening device file */
      DEBUG("Serial-con: Error, did not find CDT USB device linked to '%s' (error=%d)\n", ptInterfaceData->szInterfaceName, ptInterfaceData->iDevFile);
    }
  } else
  {
    sprintf( szDevFilePath, "/dev/%s%d", ptInterfaceData->szDevInternalName, ptInterfaceData->iDeviceNum);
    if (0 <= (ptInterfaceData->iDevFile = open( szDevFilePath, O_RDWR, O_NONBLOCK)))
    {
      /* store old configuration and setup for netX */
      tcgetattr( ptInterfaceData->iDevFile, &ptInterfaceData->old_config);
      /* RS232: configure interface */
      ConfigureIF( ptInterfaceData->iDevFile);
    } else
    {
      /* error opening device file */
      DEBUG("Serial-con: Error opening device file %s (ret=%i)\n", szDevFilePath, ptInterfaceData->iDevFile);
    }
  }
  if (0 <= ptInterfaceData->iDevFile)
  {
    /* set receive callback data */
    ptInterfaceData->pfnRxCallback     = pfnReceiveData;
    ptInterfaceData->pvRxCallbackData  = pvUser;

    pthread_attr_setstacksize( &polling_thread_attr, PTHREAD_STACK_MIN);
    ptInterfaceData->fThreadRunning = 1;

    /* create thread for read processing */
    if ((iTmpRet = pthread_create( &ptInterfaceData->tRXThread, &polling_thread_attr, CommThread, ptInterfaceData)))
    {
      DEBUG( "Serial-con: Error createing Com-Thread (Error=%d)\n", iTmpRet);
      iRet = -1;
    } else
    {
      iRet = 0;
      /* set interface state to running */
      ptInterfaceData->ulState = eINTERFACE_STATE_RUNNING;
    }
  } else
  {
    iRet = -1;
  }
  return iRet;
}

/*****************************************************************************/
/*! Stops interface and de-initializes connection
*   \param pvInterface  Pointer to interface info structure
*   \return 0 on success                                                     */
/*****************************************************************************/
int32_t APIENTRY Serial_IF_Stop( void* pvInterface)
{
  PSER_IF_DATA_T ptInterfaceData = (PSER_IF_DATA_T)pvInterface;

  if (ptInterfaceData->pvConnectorLock == NULL)
    return 0;

  OS_EnterLock( ptInterfaceData->pvConnectorLock);
  if (ptInterfaceData->ulState != eINTERFACE_STATE_STOPPED)
  {
    /* stop rX-Trhead */
    ptInterfaceData->fThreadRunning = 0;
    if (ptInterfaceData->pvRXThreadStopped != NULL) {
      /* wait for terminating thread */
      if (OS_EVENT_SIGNALLED == OS_WaitEvent(ptInterfaceData->pvRXThreadStopped, 1000)) {
        pthread_join( ptInterfaceData->tRXThread, NULL);
      } else {
        /* cancel only if thread does not stop (NOTE: cancel will not cleanup resources) */
        pthread_cancel( ptInterfaceData->tRXThread);
      }
    }
    if (ptInterfaceData->iDevFile > 0)
    {
      char tmpName[512];

      sprintf( tmpName, "%s%d", ptInterfaceData->szDevInternalName, ptInterfaceData->iDeviceNum);
      /* restore old configuration */
      if (0 == strncmp("ttyS", tmpName, strlen(tmpName)))
        tcsetattr( ptInterfaceData->iDevFile,TCSANOW,&ptInterfaceData->old_config);
      close( ptInterfaceData->iDevFile);
      ptInterfaceData->iDevFile = -1;
    }

    /* Set interface state to stopped */
    ptInterfaceData->ulState = eINTERFACE_STATE_STOPPED;
  }
  OS_LeaveLock( ptInterfaceData->pvConnectorLock);

  return NXT_NO_ERROR;
}

/*****************************************************************************/
/*! Returns interface information (currently not used)
*   \param eCmd             Command to execute
*   \param ulSize           Size of buffer pvConnectorInfo
*   \param pvInterfaceInfo  Pointer to buffer
*   \return 0 on success                                                     */
/*****************************************************************************/
int32_t APIENTRY Serial_IF_GetInformation( void* pvInterface, NETX_INTERFACE_INFO_E eCmd, uint32_t ulSize, void* pvInterfaceInfo)
{
  int32_t lRet = NXT_NO_ERROR;

  UNREFERENCED_PARAMETER(pvInterface);
  UNREFERENCED_PARAMETER(eCmd);
  UNREFERENCED_PARAMETER(ulSize);
  UNREFERENCED_PARAMETER(pvInterfaceInfo);

  return lRet;
}

/*****************************************************************************/
/*! Function validates if a serial device is connected to the interface
*   \param iDevNo Index of the interface to check (/dev/ttyS[x])
*   \return 1 if device is present                                           */
/*****************************************************************************/
int ValidateRS232Device( int iDevNo)
{
  int   fDeviceFound = 0;
  char* szValue      = NULL;
  char  szDevPath[256];

  sprintf( szDevPath, "/sys/class/tty/ttyS%d/device/uevent", iDevNo);

  /* TODO: validate vendor device product id */
  if (GetConfigString( szDevPath, "DRIVER=", &szValue))
  {
    if (0 == strncmp("serial", szValue, 6))
      fDeviceFound = 1;
  }

  if (szValue)
    free(szValue);

  if (fDeviceFound)
    return 1;
  else
    return 0;
}

/*****************************************************************************/
/*! Function validates if a netX usb device is connected to the interface
*   \param iDevNo Index of the interface to check (/dev/ttyACM[x])
*   \return 1 if device is present                                           */
/*****************************************************************************/
int ValidateUSBDevice( int iDevNo)
{
  int      fDeviceFound = 0;
  char*    szValue      = NULL;
  char     szDevPath[256];
  uint32_t ulVendor     = 0;
  uint32_t ulProduct    = 0;
  uint32_t ulDevClass   = 0;

  sprintf( szDevPath, "/sys/class/tty/ttyACM%d/device/uevent", iDevNo);

  /* validate vendor device product id       */
  /* TOOD: add missing product and class ids */
  if (GetConfigString( szDevPath, "PRODUCT=", &szValue))
  {
    if (3 == sscanf( szValue, "%X/%X/%X", &ulVendor, &ulProduct, &ulDevClass))
    {
      switch (ulVendor)
      {
        case HILSCHER_VENDOR_ID:
          /* say yes to all products */
          fDeviceFound = 1;
        break;
        default:
        break;
      }
    }
  }
  if (szValue)
    free(szValue);

  if (fDeviceFound)
    return 1;
  else
    return 0;
}

/*****************************************************************************/
/*! Reads and validates the serial configuration file ("serialconfig")
*   \param ppptNamelist Pointer to an array of interfaces
*   \return number interfaces ppptNamelist                                   */
/*****************************************************************************/
uint32_t ReadConfig(struct dirent*** ppptNamelist)
{
  char*           szValue     = NULL;
  uint32_t        ulConfCount = 0;
  char            szInterface[10];
  struct dirent** pptNamelist = NULL;

  do
  {
    sprintf( szInterface, "IF%d=", ulConfCount);

    if (szValue)
      free( szValue);

    szValue = NULL;
    if (GetConfigString( "./serialconfig", szInterface, &szValue))
    {
      void* tmp = realloc( pptNamelist, sizeof(struct dirent*)*ulConfCount+1);

      if (tmp != NULL) {
        pptNamelist = tmp;
        pptNamelist[ulConfCount] = malloc(sizeof(struct dirent));
        sprintf( pptNamelist[ulConfCount]->d_name, "%s", szValue);
        ulConfCount++;
      } else {
        fprintf(stderr, "Serial-con: Error allocating enough memory for configuration!\n");
      }
    } else
    {
      break;
    }
  } while( (szValue != NULL) && (ulConfCount<100));

  pptNamelist = realloc( pptNamelist, sizeof(struct dirent*)*ulConfCount+1);
  pptNamelist[ulConfCount] = NULL;

  *ppptNamelist = pptNamelist;

  return ulConfCount;
}


/*****************************************************************************/
/*! Iterates over all interfaces and verifies whether a netX is connected to.
*   Every detected device is stored in a global list.                        */
/*****************************************************************************/
static void DetectNetxDevice( void)
{
  struct dirent**  pptNamelist     = NULL;
  int              iSerDevCount    = 0;
  int              iInterfaceIndex = 0;

  /* check if a configuration files exists */
  if (0 == (iSerDevCount = ReadConfig( &pptNamelist)))
  {
    /* there is no configuration file so iterate over all serial interfaces */
    iSerDevCount = scandir("/sys/class/tty", &pptNamelist, 0, alphasort);
  }
  if(iSerDevCount > 0)
  {
    int iCurrentSer  = 0;
    int fDeviceFound = 0;

    /* check every interface if a netX device is connected to */
    for(iCurrentSer = 0; iCurrentSer < iSerDevCount; ++iCurrentSer)
    {
      int iDevNo;

      /* check for usb connection */
      if (0 == strncmp( "ttyACM", pptNamelist[iCurrentSer]->d_name, 6))
      {
        if(1 == sscanf(pptNamelist[iCurrentSer]->d_name,
                        "ttyACM%u",
                        &iDevNo))
        {
          /* check if netX is connected */
          if( ValidateUSBDevice( iDevNo))
            fDeviceFound = 1;

        }
      /* check RS232 connection */
      } else if (0 == strncmp( "ttyS", pptNamelist[iCurrentSer]->d_name, 4))
      {
        if(1 == sscanf(pptNamelist[iCurrentSer]->d_name,
                      "ttyS%u",
                      &iDevNo))
        {
          /* check if netX is connected */
          if (ValidateRS232Device( iDevNo))
            fDeviceFound = 1;
        }
      } else
      {
        /* device not known */
      }

      if (fDeviceFound)
      {
        SER_IF_DATA_T* ptInterfaceData = NULL;
        char           szDeviceName[16];
        char           szTmpPath[278];

        /* we have found a netX */
        snprintf( szDeviceName, 15, "COM%d", s_tSerConnectorInfo.ulConfigCount++);

        ptInterfaceData = malloc(sizeof(SER_IF_DATA_T));

        memset( ptInterfaceData, 0, sizeof(SER_IF_DATA_T));
        ptInterfaceData->iDevFile = -1;

        /* store device path to be able to identify port */
        snprintf( szTmpPath, 278,"/sys/class/tty/%s", pptNamelist[iCurrentSer]->d_name);
        realpath( szTmpPath, ptInterfaceData->szPortIdent);      /* Interface identification string, required to identify the connected device.       */
                                                                 /* A usb device will change its interface name if it becomes removed and plugged in. */
                                                                 /* (enables usb hotplug)                                                             */
        strcpy( ptInterfaceData->szInterfaceName, szDeviceName); /* interface alias */

        /* store device file */
        sscanf(pptNamelist[iCurrentSer]->d_name, "%[^0-9]s", ptInterfaceData->szDevInternalName);
        sscanf(pptNamelist[iCurrentSer]->d_name, "%*[^0-9]%d", &ptInterfaceData->iDeviceNum);

        TAILQ_INSERT_TAIL(&s_tSerConnectorInfo.tConnectorList, ptInterfaceData, tList);
        fDeviceFound  = 0;
      }
    }
  }
  while (iSerDevCount>0) {
    iSerDevCount--;
    free(pptNamelist[iSerDevCount]);
  }
  if (NULL != pptNamelist)
    free( pptNamelist);
}

/*****************************************************************************/
/*! Parses connector configuration file
*   \param szFile   File name
*   \param szKey    Key to search for
*   \param szValue  Returned value (if key was found)
*   \return != 0 on success                                                  */
/*****************************************************************************/
static int GetConfigString(const char* szFile, const char* szKey, char** szValue)
{
  int   ret = 0;
  FILE* fd  = fopen(szFile, "r");

  if(NULL != fd)
  {
    /* File is open */
    char* buffer = malloc(PARSER_BUFFER_SIZE);

    /* Read file line by line */
    while(NULL != fgets(buffer, PARSER_BUFFER_SIZE, fd))
    {
      char* key;

      /* '#' marks a comment line in the device.conf file */
      if(buffer[0] == '#')
        continue;

      /* Search for key in the input buffer */
      key = (char*)strcasestr(buffer, szKey);

      if(NULL != key)
      {
        /* We've found the key */
        int   allocsize  = strlen(key + strlen(szKey)) + 1;
        int   valuelen;
        char* tempstring = (char*)malloc(allocsize);

        strcpy(tempstring, key + strlen(szKey));
        valuelen = strlen(tempstring);

        /* strip all trailing whitespaces */
        while( (tempstring[valuelen - 1] == '\n') ||
               (tempstring[valuelen - 1] == '\r') ||
               (tempstring[valuelen - 1] == ' ') )
        {
          tempstring[valuelen - 1] = '\0';
          --valuelen;
        }

        *szValue = tempstring;
        ret = 1;
        break;
      }
    }

    free(buffer);
    fclose(fd);
  }

  return ret;
}

/*****************************************************************************/
/*! Sets connector configuration
*   \param fd file descriptor to serial com port
*   \return 0 on success                                                     */
/*****************************************************************************/
int ConfigureIF( int fd) {
  struct termios tTermios;

  memset(&tTermios,0,sizeof(tTermios));
  /* store old configuration and setup for netX */
  tTermios.c_iflag = 0;
  tTermios.c_oflag = 0;
  tTermios.c_cflag = CS8|CREAD|CLOCAL;
  cfsetospeed( &tTermios, B115200);
  cfsetispeed( &tTermios, B115200);

  return tcsetattr( fd, TCSANOW, &tTermios);
}
